# include "hugin.h"

# include <stdlib.h>
# include <string.h>
# include <math.h>

/* A simple parse error handler: It prints the error message on stderr. */

void error_handler (h_location_t line_no, h_string_t message, void *data)
{
    fprintf (stderr, "Error at line %d: %s\n", (int) line_no, message);
}

/* This function is used when a Hugin API error is detected: It prints
   an error message on stderr and exits the program.
*/

void print_error (void)
{
    fprintf (stderr, "Error: %s\n", h_error_description (h_error_code ()));
    exit (EXIT_FAILURE);
}


void print_junction_trees (h_domain_t);
void print_beliefs_and_utilities (h_domain_t);

int domain_has_utilities (h_domain_t);


/* This function parses the given NET file, compiles the network, and
   prints the prior beliefs and expected utilities of all nodes.  If a
   case file is given, the function loads the file, propagates the
   evidence, and prints the updated results.

   If the network is a LIMID, we assume that we should compute
   policies for all decisions (rather than use the ones specified in
   the NET file).  Likewise, we update the policies when new evidence
   arrives.
*/

void load_and_propagate (h_string_t net_file_name, h_string_t case_file_name)
{
    size_t l = strlen (net_file_name);
    char *file_name_buffer;
    h_domain_t domain;
    FILE *log_file;

    if (l >= 4 && strcmp (net_file_name + (l - 4), ".net") == 0)
	l -= 4;

    if ((file_name_buffer = malloc (l + 5)) == NULL)
    {
	fprintf (stderr, "Out of memory\n");
	exit (EXIT_FAILURE);
    }

    strcpy (file_name_buffer, net_file_name);
    strcpy (file_name_buffer + l, ".net");

    if ((domain = h_net_parse_domain (file_name_buffer, error_handler, NULL))
		== NULL)
	print_error ();

    strcpy (file_name_buffer + l, ".log");

    if ((log_file = fopen (file_name_buffer, "w")) == NULL)
    {
	fprintf (stderr, "Could not open \"%s\"\n", file_name_buffer);
	exit (EXIT_FAILURE);
    }

    free (file_name_buffer);

    h_domain_set_log_file (domain, log_file);

    if (h_domain_triangulate (domain, h_tm_best_greedy) != 0)
	print_error ();

    if (h_domain_compile (domain) != 0)
	print_error ();

    h_domain_set_log_file (domain, NULL);

    fclose (log_file);

    print_junction_trees (domain);

    if (!domain_has_utilities (domain))
	printf ("\n\nPrior beliefs:\n");
    else
    {
	if (h_domain_update_policies (domain) != 0)
	    print_error ();

	printf ("\n\nOverall expected utility: %g\n",
		h_domain_get_expected_utility (domain));

	printf ("\nPrior beliefs (and expected utilities):\n");
    }

    print_beliefs_and_utilities (domain);

    if (case_file_name != NULL)
    {
	if (h_domain_parse_case (domain, case_file_name, error_handler, NULL)
		!= 0)
	    print_error ();

	printf ("\n\nPropagating the evidence specified in \"%s\"\n",
		case_file_name);

	if (h_domain_propagate (domain, h_equilibrium_sum, h_mode_normal) != 0)
	    print_error ();

	printf ("\nP(evidence) = %g\n",
		h_domain_get_normalization_constant (domain));

	if (!domain_has_utilities (domain))
	    printf ("\nUpdated beliefs:\n");
	else
	{
	    if (h_domain_update_policies (domain) != 0)
		print_error ();

	    printf ("\nOverall expected utility: %g\n",
		    h_domain_get_expected_utility (domain));

	    printf ("\nUpdated beliefs (and expected utilities):\n");
	}

	print_beliefs_and_utilities (domain);
    }

    h_domain_delete (domain);
}

void print_nodes (h_node_t *);

/* Print the cliques of the junction tree(s). */

void print_junction_trees (h_domain_t domain)
{
    h_junction_tree_t jt = h_domain_get_first_junction_tree (domain);

    if (jt == NULL)
	print_error ();

    for (; jt != NULL; jt = h_jt_get_next (jt))
    {
	h_clique_t *list = h_jt_get_cliques (jt);

	if (list == NULL)
	    print_error ();

	printf ("Junction tree:\n");

	for (; *list != NULL; list++)
	{
	    h_node_t *members = h_clique_get_members (*list);

	    if (members == NULL)
		print_error ();

	    printf ("  Clique:");
	    print_nodes (members);
	} 
    }
}


/* Print the beliefs and expected utilities of all nodes in the domain. */

void print_beliefs_and_utilities (h_domain_t domain)
{
    int has_utilities = domain_has_utilities (domain);
    h_node_t node = h_domain_get_first_node (domain);

    for (; node != NULL; node = h_node_get_next (node))
    {
	h_node_category_t category = h_node_get_category (node);
	char type = (category == h_category_chance ? 'C'
		     : category == h_category_decision ? 'D'
		     : category == h_category_utility ? 'U' : 'F');

	printf ("\n[%c] %s (%s)\n", type, h_node_get_label (node),
		h_node_get_name (node));

	if (category == h_category_utility)
	    printf ("  - Expected utility: %g\n",
		    h_node_get_expected_utility (node, 0));
	else if (category == h_category_function)
	{
	    h_double_t value = h_node_get_value (node);

	    if (h_error_code () != 0)
		printf ("  - Value: N/A\n");
	    else
		printf ("  - Value: %g\n", value);
	}
	else if (h_node_get_kind (node) == h_kind_discrete)
	{
	    h_count_t state_count = h_node_get_number_of_states (node);
	    size_t i;

	    if (has_utilities)
		for (i = 0; i < state_count; i++)
		    printf ("  - %s %g (%g)\n",
			    h_node_get_state_label (node, i),
			    h_node_get_belief (node, i),
			    h_node_get_expected_utility (node, i));
	    else
		for (i = 0; i < state_count; i++)
		    printf ("  - %s %g\n",
			    h_node_get_state_label (node, i),
			    h_node_get_belief (node, i));
	}
	else  /* "node" is a continuous chance node */
	{
	    printf ("  - Mean : %g\n", h_node_get_mean (node));
	    printf ("  - SD   : %g\n", sqrt (h_node_get_variance (node)));
	}
    }
}


/* Print the names of all nodes in the list. */

void print_nodes (h_node_t *list)
{
    for (; *list != NULL; list++)
    {
	h_string_t name = h_node_get_name (*list);

	if (name == NULL)
	    print_error ();

	printf (" %s", name);
    }

    printf ("\n");
}


/* Are there utility nodes in "domain"? */

int domain_has_utilities (h_domain_t domain)
{
    h_node_t node = h_domain_get_first_node (domain);

    for (; node != NULL; node = h_node_get_next (node))
	if (h_node_get_category (node) == h_category_utility)
	    return 1;

    return 0;
}


/*
 * Load a Hugin NET file, compile the network, and print the results.
 * If a case file is specified, load it, propagate the evidence, and
 * print the results.
 */

int main (int argc, char *argv[])
{
    if (argc < 2 || argc > 3)
    {
	fprintf (stderr, "Usage: %s <NET_file_name> [<case_file_name>]\n",
		 argv[0]);
	exit (EXIT_FAILURE);
    }

    load_and_propagate (argv[1], argv[2]);

    return 0;
}
